//
//  pe_mtl_templates.metal
//  libpaintengine
//
//  Created by Georgy Ostrobrod on 05/02/16.
//  Copyright © 2016 Pixelmator. All rights reserved.
//

#include <metal_stdlib>
using namespace metal;

////////////////////////////////////////////////////////////////////////////////
// MARK: **************************** Merge ************************************



// ------------------------------  Types ---------------------------------------

struct VertexIO
{
    float4 m_Position [[position]];
    float2 m_TexCoord [[user(texturecoord)]];
};



struct Vertex2IO
{
    float4 m_Position  [[position]];
    float2 m_Tex1Coord [[user(texture1coord)]];
    float2 m_Tex2Coord [[user(texture2coord)]];
};



struct MergeTemplateOptions {
    int m_Options        [[user(options)]];
    int m_AlphaIndex     [[user(alphaindex)]];
    int m_MaskAlphaIndex [[user(maskalphaindex)]];
    int m_BlendTypeInner [[user(blendtypeinner)]];
    float4 m_Color       [[user(color)]];
    float m_OpacityLimit [[user(opacitylimit)]];
    float m_OpacityLayer [[user(opacitylayer)]];
    float m_Reserved0      [[user(reserved0)]];
    float m_Reserved1      [[user(reserved1)]];
};



enum FragmentMergeOptions {
    MERGE_OPTIONS_NONE       = 0,
    MERGE_OPTIONS_ERASER     = 1,
    MERGE_OPTIONS_SRGB       = MERGE_OPTIONS_ERASER   << 1,
    MERGE_OPTIONS_1CHANNEL   = MERGE_OPTIONS_SRGB     << 1,
    MERGE_OPTIONS_2CHANNEL   = MERGE_OPTIONS_1CHANNEL << 1,
    MERGE_OPTIONS_OUT_CANVAS = MERGE_OPTIONS_2CHANNEL << 1,
    MERGE_OPTIONS_N_CH_MASK   = MERGE_OPTIONS_OUT_CANVAS << 1
};
    
    
template<typename T>
vec<T, 4> post_process(vec<T, 4> src,
                       vec<T, 4> dst,
                       int alphaIndex,
                       float2 curPos,
                       T opacity);



// ------------------------------ Fragment -------------------------------------

// Simple texture applying
vertex VertexIO vSimple(
    device float4 *pPosition   [[ buffer(0) ]],
    device float2 *pTexCoords  [[ buffer(1) ]],
    uint           vid         [[ vertex_id ]])
{
    VertexIO outVertices;
    float4 pos = pPosition[vid];
    pos.y = -pos.y;
    outVertices.m_Position = pos;
    outVertices.m_TexCoord = pTexCoords[vid];
    
    return outVertices;
}



vertex Vertex2IO vSimple2(
    device float4 *pPosition   [[ buffer(0) ]],
    device float2 *pTex1Coords [[ buffer(1) ]],
    device float2 *pTex2Coords [[ buffer(2) ]],
    uint           vid         [[ vertex_id ]])
{
    Vertex2IO outVertices;
    
    float4 pos = pPosition[vid];
    pos.y = -pos.y;
    outVertices.m_Position = pos;
    outVertices.m_Tex1Coord = pTex1Coords[vid];
    outVertices.m_Tex2Coord = pTex2Coords[vid];
    
    return outVertices;
}



// ------------------------------ Fragment -------------------------------------

fragment half4 fMergeInplaceTemplate(
    VertexIO                        inFrag     [[ stage_in  ]],
    texture2d<half, access::sample> srcTexture [[ texture(0)]],
#if PTP_MTL_IOS || PTP_MTL_APPLE_SILICON
    half4        dstColor [[ color(0) ]],
#else
    texture2d<half, access::sample> bkgTexture [[ texture(2) ]],
#endif
    constant MergeTemplateOptions   *options   [[ buffer(0)  ]])
{
    constexpr sampler quadSampler(address::clamp_to_zero, filter::nearest);
    half4 src = srcTexture.sample(quadSampler, inFrag.m_TexCoord);
    
#if PTP_MTL_OSX && !PTP_MTL_APPLE_SILICON
    half4 dstColor = bkgTexture.sample(quadSampler, inFrag.m_TexCoord);
#endif
    
    return post_process(src,
                        dstColor,
                        options->m_AlphaIndex,
                        inFrag.m_Position.xy,
                        half(options->m_OpacityLayer));
}
    
    
    
fragment half4 fMergeSimple2Template(
    Vertex2IO                       inFrag     [[ stage_in   ]],
    texture2d<half, access::sample> srcTexture [[ texture(0) ]],
    texture2d<half, access::sample> dstTexture [[ texture(1) ]],
    sampler srcSampler [[ sampler(0) ]],
    sampler dstSampler [[ sampler(1) ]],
#if PTP_MTL_IOS || PTP_MTL_APPLE_SILICON
    half4        dstColor [[ color(0) ]],
#else
    texture2d<half, access::sample> bkgTexture [[ texture(2) ]],
#endif
    constant MergeTemplateOptions  *options    [[ buffer(0) ]])
{
    half4 dst = dstTexture.sample(srcSampler, inFrag.m_Tex2Coord);
    half4 src = srcTexture.sample(dstSampler, inFrag.m_Tex1Coord);
    
    half4 src0 = 0.0;
    if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
        src = src.rrrr;
    }
    else if (options->m_Options & MERGE_OPTIONS_2CHANNEL) {
        if (options->m_Options & MERGE_OPTIONS_N_CH_MASK) {
            Scalar a = src[options->m_MaskAlphaIndex];
            src = Vector(options->m_Color.r * a, a, a, a);
        }
        else {
            src.r *= src.g;
        }
        src.ba = src.gg;
    }
    else {
        if (options->m_Options & MERGE_OPTIONS_N_CH_MASK) {
            Scalar a = src[options->m_MaskAlphaIndex];
            src = Vector(options->m_Color.rgb * a, a);
        }
        else {
            src.rgb *= src.a;
        }
    }
    
    if (options->m_Options & MERGE_OPTIONS_ERASER) {
        src0 = dst * (1.0 - src[options->m_AlphaIndex]);
    }
    else {
        int alphaIndex = options->m_AlphaIndex;
        if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
            src.r *= options->m_Color.r;
            alphaIndex = 1;
            dst = dst.rrrr;
        }

        src0 = blendCC(src,
                       dst,
                       alphaIndex,
                       inFrag.m_Position.xy,
                       options->m_BlendTypeInner,
                       options->m_Options & MERGE_OPTIONS_SRGB);
        
        if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
            src0 = src0.rrrr;
        }
    }
    
#if PTP_MTL_OSX && !PTP_MTL_APPLE_SILICON
    half4 dstColor = bkgTexture.sample(dstSampler, inFrag.m_Tex2Coord);
#endif
    
//    if (options->m_Options & MERGE_OPTIONS_OUT_CANVAS) {
//        src0 *= src.a;
//    }
    
    return post_process(src0,
                        dstColor,
                        options->m_AlphaIndex,
                        inFrag.m_Position.xy,
                        half(options->m_OpacityLayer));
}
    
    
    
fragment half4 fMergeSimpleTemplate(
    VertexIO                        inFrag     [[ stage_in  ]],
    texture2d<half, access::sample> srcTexture [[ texture(0)]],
    sampler srcSampler [[ sampler(0) ]],
#if PTP_MTL_IOS || PTP_MTL_APPLE_SILICON
    half4        dstColor [[ color(0) ]],
#else
    texture2d<half, access::sample> bkgTexture [[ texture(2) ]],
    sampler dstSampler [[ sampler(1) ]],
#endif
    constant MergeTemplateOptions   *options   [[ buffer(0)  ]])
{
    half4 src = srcTexture.sample(srcSampler, inFrag.m_TexCoord);
    half4 dst = 0.0;
    
    half4 src0 = 0.0;
    if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
        src = src.rrrr;
    }
    else if (options->m_Options & MERGE_OPTIONS_2CHANNEL) {
        if (options->m_Options & MERGE_OPTIONS_N_CH_MASK) {
            Scalar a = src[options->m_MaskAlphaIndex];
            src = Vector(options->m_Color.r * a, a, a, a);
        }
        else {
            src.r *= src.g;
        }
        src.ba = src.gg;
    }
    else {
        if (options->m_Options & MERGE_OPTIONS_N_CH_MASK) {
            Scalar a = src[options->m_MaskAlphaIndex];
            src = Vector(options->m_Color.rgb * a, a);
        }
        else {
            src.rgb *= src.a;
        }
    }
    
    if (options->m_Options & MERGE_OPTIONS_ERASER) {
        src0 = dst * (1.0 - src[options->m_AlphaIndex]);
    }
    else {
        int alphaIndex = options->m_AlphaIndex;
        if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
            src.r *= options->m_Color.r;
            alphaIndex = 1;
            dst = dst.rrrr;
        }

        src0 = blendCC(src,
                       dst,
                       alphaIndex,
                       inFrag.m_Position.xy,
                       options->m_BlendTypeInner,
                       options->m_Options & MERGE_OPTIONS_SRGB);
        
        if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
            src0 = src0.rrrr;
        }
    }
    
#if PTP_MTL_OSX && !PTP_MTL_APPLE_SILICON
    half4 dstColor = bkgTexture.sample(dstSampler, inFrag.m_TexCoord);
#endif
    
//    if (options->m_Options & MERGE_OPTIONS_OUT_CANVAS) {
//        src0 *= src.a;
//    }
    
    return post_process(src0,
                        dstColor,
                        options->m_AlphaIndex,
                        inFrag.m_Position.xy,
                        half(options->m_OpacityLayer));
}



fragment half4 fMergeWatercolor2Template(
    Vertex2IO          inFrag   [[ stage_in  ]],
    texture2d<half, access::sample> srcTexture [[ texture(0) ]],
    texture2d<half, access::sample> dstTexture [[ texture(1) ]],
    texture2d<half, access::sample> lutTexture [[ texture(3) ]],
    sampler srcSampler [[ sampler(0) ]],
    sampler dstSampler [[ sampler(1) ]],
    sampler lutSampler [[ sampler(2) ]],
#if PTP_MTL_IOS || PTP_MTL_APPLE_SILICON
    half4        dstColor [[ color(0) ]],
#else
    texture2d<half, access::sample> bkgTexture [[ texture(2) ]],
#endif
    constant MergeTemplateOptions *options     [[ buffer(0)  ]])
{
    half4 dst = dstTexture.sample(srcSampler, inFrag.m_Tex2Coord);
    half4 src = srcTexture.sample(dstSampler, inFrag.m_Tex1Coord);
    
    half a = min(src.r, src.g);
    a = options->m_OpacityLimit * lutTexture.sample(lutSampler, float2(a, 0.5)).r;
    
    // Set source solor.
    src = half4(half3(options->m_Color.rgb), a);
    
    half4 src0 = 0.0;
    if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
        src = src.rrrr;
    }
    else if (options->m_Options & MERGE_OPTIONS_2CHANNEL) {
        src.r *= src.g;
        src.ba = src.gg;
    }
    else {
        src.rgb *= src.a;
    }
    
    if (options->m_Options & MERGE_OPTIONS_ERASER) {
        src0 = dst * (1.0 - src[options->m_AlphaIndex]);
    }
    else {
        int alphaIndex = options->m_AlphaIndex;
        if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
            src.r *= options->m_Color.r;
            alphaIndex = 1;
            dst = dst.rrrr;
        }

        src0 = blendCC(src,
                       dst,
                       alphaIndex,
                       inFrag.m_Position.xy,
                       options->m_BlendTypeInner,
                       options->m_Options & MERGE_OPTIONS_SRGB);
        
        if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
            src0 = src0.rrrr;
        }
    }
    
#if PTP_MTL_OSX && !PTP_MTL_APPLE_SILICON
    half4 dstColor = bkgTexture.sample(dstSampler, inFrag.m_Tex2Coord);
#endif
    
//    if (options->m_Options & MERGE_OPTIONS_OUT_CANVAS) {
//        src0 *= src.a;
//    }

    return post_process(src0,
                        dstColor,
                        options->m_AlphaIndex,
                        inFrag.m_Position.xy,
                        half(options->m_OpacityLayer));
}
    
    
    
fragment half4 fMergeWatercolorTemplate(
    VertexIO                        inFrag     [[ stage_in   ]],
    texture2d<half, access::sample> srcTexture [[ texture(0) ]],
    texture2d<half, access::sample> lutTexture [[ texture(3) ]],
    sampler srcSampler [[ sampler(0) ]],
    sampler lutSampler [[ sampler(2) ]],
#if PTP_MTL_IOS || PTP_MTL_APPLE_SILICON
    half4        dstColor [[ color(0) ]],
#else
    texture2d<half, access::sample> bkgTexture [[ texture(2) ]],
    sampler dstSampler [[ sampler(1) ]],
#endif
    constant MergeTemplateOptions *options     [[ buffer(0)  ]])
{
    half4 src = srcTexture.sample(srcSampler, inFrag.m_TexCoord);
    half4 dst = 0.0;
    
    half a = min(src.r, src.g);
    a = options->m_OpacityLimit * lutTexture.sample(lutSampler, float2(a, 0.5)).r;
    
    // Set source solor.
    src = half4(half3(options->m_Color.rgb), a);
    
    half4 src0 = 0.0;
    if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
        src = src.rrrr;
    }
    else if (options->m_Options & MERGE_OPTIONS_2CHANNEL) {
        src.r *= src.g;
        src.ba = src.gg;
    }
    else {
        src.rgb *= src.a;
    }
    
    if (options->m_Options & MERGE_OPTIONS_ERASER) {
        src0 = dst * (1.0 - src[options->m_AlphaIndex]);
    }
    else {
        int alphaIndex = options->m_AlphaIndex;
        if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
            src.r *= options->m_Color.r;
            alphaIndex = 1;
            dst = dst.rrrr;
        }

        src0 = blendCC(src,
                       dst,
                       alphaIndex,
                       inFrag.m_Position.xy,
                       options->m_BlendTypeInner,
                       options->m_Options & MERGE_OPTIONS_SRGB);
        
        if (options->m_Options & MERGE_OPTIONS_1CHANNEL) {
            src0 = src0.rrrr;
        }
    }
    
#if PTP_MTL_OSX && !PTP_MTL_APPLE_SILICON
    half4 dstColor = bkgTexture.sample(dstSampler, inFrag.m_TexCoord);
#endif
    
//    if (options->m_Options & MERGE_OPTIONS_OUT_CANVAS) {
//        src0 *= src.a;
//    }
    
    return post_process(src0,
                        dstColor,
                        options->m_AlphaIndex,
                        inFrag.m_Position.xy,
                        half(options->m_OpacityLayer));
}

// Add code of post_process here:
// Example (src return without post porcessing):
/*
 template<typename T>
 vec<T, 4> post_process(vec<T, 4> src,
                        vec<T, 4> dst,
                        int alphaIndex,
                        float2 curPos,
                        T opacity)
 {
    return src;
 }
 */
